//
//  TNKDriveNavDelegate.h
//  TencentNavKit
//
//  Created by mol on 2022/10/9.
//  Copyright © 2022 Tencent. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <TencentNavKit/TNKRerouteCommon.h>

NS_ASSUME_NONNULL_BEGIN

@class TNKDriveNavManager,
TNKLocation,
TNKGpsStatus,
TNKDriveNavRoute,
TNKDriveRoute,
TNKDriveNavigationData,
TNKDriveRerouteResult,
TNKLaneInfo,
TNKParallelRoadStatus,
TNKChangeRouteMsg,
TNKIdleSectionInfo,
TNKRecommendRouteInfo,
TNKNavToWayPointInfo,
TNKTrafficJamInfo,
TNKTollStationInfo,
TNKNavCameraInfo,
TNKNavStartFailInfo,
TNKNavRestrictionInfo;

/// 驾车导航事件回调
@protocol TNKDriveNavDelegate <NSObject>

@optional
/**
 * @brief 导航开始.
 * @param manager manager
 */
- (void)driveNavManagerDidStart:(TNKDriveNavManager *)manager;

/**
 * @brief 导航结束.
 * @param manager manager
 */
- (void)driveNavManagerDidStop:(TNKDriveNavManager *)manager;

/**
 * @brief slice状态下导航开启的回调，根据success自行处理后续的逻辑，如失败重新算路等;如三段slice的情况下自行在success的情况下发起导航
 * @param nav_session_id session id
 * @param success 成功或者失败
 */
- (void)driveNavManagerOnNavigationDataReady:(TNKDriveNavManager *)manager
                          nav_session_id:(NSString *)nav_session_id
                                 success:(BOOL)success;

/**
 * @brief 获取到达终点附近的回调. 尚未结束导航时, 这个回调可能执行多次.
 * @param manager manager
 */
- (void)driveNavManagerWillArriveDestination:(TNKDriveNavManager *)manager;

/**
 * @brief 获取导航面板信息
 * @param manager manager
 * @param navigationData 导航面板信息
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager updateNavigationData:(TNKDriveNavigationData *)navigationData;

/**
 * @brief 获取导航定位更新回调.
 * @param manager manager
 * @param location 定位信息
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager
      didUpdateLocation:(TNKLocation *)location;

/**
 * @brief 获取定位失败的回调
 * @param manager manager
 * @param error 错误信息
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager updateLocationFailedWithError:(NSError *)error;

/**
 * @brief 获取GPS信号状态变化的回调.
 * @param manager manager
 * @param gpsStatus gps信号状态
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager gpsStatusDidChange:(TNKGpsStatus *)gpsStatus;


/**
 * @brief 获取重新算路请求开始回调.
 * @param manager manager
 * @param rerouteReason 重新算路类型
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager didStartRerouting:(TNKRerouteReason)rerouteReason;

/**
 * @brief 导航开始失败回调.
 * @param manager manager
 * @param startFailInfo 导航开始失败回调内容
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager didStartNavFail:(TNKNavStartFailInfo *)startFailInfo;

/**
 * @brief 获取重新算路请求成功回调.
 * @param manager manager
 * @param rerouteresult 重新算路成功结果
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager
      rerouteDidSuccess:(TNKDriveRerouteResult *)rerouteresult;

/**
 * @brief 获取重新算路请求失败回调.
 * @param manager manager
 * @param rerouteresult 重新算路成功结果
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager
         rerouteDidFail:(TNKDriveRerouteResult *)rerouteresult;

/**
 * @brief 获取重新算路请求的起点命中无路网回调。如果新的起点命中无路网的位置，则暂时不下发新路线
 * @param manager manager
 * @param rerouteReason 偏航的原因
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager
      rerouteDidInFence:(TNKRerouteReason)rerouteReason;

/**
 * @brief 更新路线路况数据回调. 具体信息可通过manager.curNavRoute.curTrafficItems取得最新的路况数据；通过manager.backupNavRoutes获取伴随路线的路况信息
 *
 * @param manager manager
 */
- (void)driveNavManagerDidUpdateRotueTraffic:(TNKDriveNavManager *)manager;

/**
 * @brief 获取切换平行路状态更新回调.
 * @since 5.2.8
 * @param manager manager
 * @param parallelRoadStatus 切换平行路状态
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager updateParallelRoadStatus:(TNKParallelRoadStatus *)parallelRoadStatus;

/**
 * @brief 显示车道信息回调.
 * @since 5.3.3
 * @param manager manager
 * @param laneInfo 车道信息回调.
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager showGuidedLaneInfo:(TNKLaneInfo *)laneInfo;

/**
 * @brief 隐藏车道线回调.
 * @param manager manager
 */
- (void)driveNavManagerHideGuidedLaneInfo:(TNKDriveNavManager *)manager;

/**
 * @brief 开启多路线功能时，删除备选路线的回调.
 * @since 5.2.6
 * @param manager manager
 * @param deletedRouteIDs 删除掉的路线ID数组
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager
  didDeleteBackupRoutes:(NSArray<NSString *> *)deletedRouteIDs;


/**
 * @brief 开启多路线功能时，补充了备选路线的回调.
 * @since 5.2.6
 * @param manager manager
 * @param backupRoutes 补充的备选路线数组
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager
     didAddBackupRoutes:(NSArray<TNKDriveRoute *> *)backupRoutes;

/**
 * @brief 路线切换回调.
 * @since 5.4.0
 * @param manager manager
 * @param navRoute 路线数据
 * @param msg 换路理由等信息
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager
         routeDidChange:(TNKDriveNavRoute *)navRoute
         changeRouteMsg:(TNKChangeRouteMsg *)msg;

/**
 * @brief 进入空闲区间的回调.
 * @since 5.4.1
 * @param manager manager
 * @param idleSectionInfo 空闲区间信息
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager didEnterIdleSection:(TNKIdleSectionInfo *)idleSectionInfo;

/**
 * @brief 推荐新路线的回调. 开发者可以根据该回调做弹框提示.
 * @since 5.4.0
 * @param manager manager
 * @param recommendRouteInfo 推荐新路线的信息
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager showRecommendedRouteInfo:(TNKRecommendRouteInfo *)recommendRouteInfo;


/**
 * @brief 获取经过途经点回调.
 * @param manager manager
 * @param waypointInfo 该途经点信息
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager
  didArriveWaypointInfo:(TNKNavToWayPointInfo *)waypointInfo;

/**
 * @brief 收费站信息更新回调. 通常在驶出收费站时回调
 * @param manager manager
 * @param tollStationInfo 所有走过的收费站信息
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager didUpdateTollStationInfo:(TNKTollStationInfo *)tollStationInfo;

/**
 * @brief 前方拥堵区域信息的更新回调。
 * @param manager manager
 * @param trafficJamInfo 前方拥堵区域信息。为nil时代表拥堵消失
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager didUpdateTrafficJamInfo:(TNKTrafficJamInfo * _Nullable)trafficJamInfo;

/**
 * @brief 前方电子眼信息，会给出一定距离内的电子眼信息。size为0时表示前方一定距离内没有电子眼
 * @param manager manager
 * @param navCameras 电子眼信息数组
 */
- (void)driveNavManager:(TNKDriveNavManager *)manager didUpdateCameraInfo:(NSArray<TNKNavCameraInfo *> *)navCameras;

@end

NS_ASSUME_NONNULL_END
