//
//  TNKDriveCommonObj.h
//  TencentNavKit
//
//  Created by mol on 2022/7/26.
//  Copyright © 2022 Tencent. All rights reserved.
//

#import <TencentNavKit/TNKCommonObj.h>
#import <TencentNavKit/TNKDriveRouteExplain.h>
#import <TencentNavKit/TNKRouteTrafficItem.h>
#import <TencentNavKit/TNKSegmentItem.h>
#import <TencentNavKit/TNKLocation.h>

NS_ASSUME_NONNULL_BEGIN

@class TNKTruckSearchOption;
@class TNKForkPoint;
@class TNKNavToWayPointInfo;
@class TencentLBSLocation;
@class TNKTruckRestrictionItem;
@class TNKTurnRestrictionItem;
@class TNKCustomRouteReason;

/**
 * @brief 路线规划的错误
 */
extern NSString * const TNKDriveRouteSearchErrorDomain;

typedef enum _TNKDriveRouteSearchErrorCode
{
    TNKDriveRouteSearchErrorCode_NetworkError                 = 1001,     ///< 网络错误
    TNKDriveRouteSearchErrorCode_NoNetwork                    = 1002,     ///< 手机无网络
    TNKDriveRouteSearchErrorCode_RequestTimeOut               = 1003,     ///< 请求超时
    TNKDriveRouteSearchErrorCode_DataInvalid                  = 2001,     ///< 返回数据无效
    TNKDriveRouteSearchErrorCode_StartDestPointError          = 2002,     ///< 起终点参数错误
    TNKDriveRouteSearchErrorCode_WayPointError                = 2003,     ///< 途经点参数错误
    TNKDriveRouteSearchErrorCode_AdsorptionFailed             = 2004,     ///< 吸附失败
    TNKDriveRouteSearchErrorCode_CalculateFailed              = 2005,     ///< 算路失败
    TNKDriveRouteSearchErrorCode_InvalidKeyError              = 2006,     ///< 鉴权失败
    TNKDriveRouteSearchErrorCode_NavigationParaInvalid        = 2007,     ///< 点串导航时参数不合法
    TNKDriveRouteSearchErrorCode_ChangeRouteByRouteIDFailed   = 2008,     ///< 乘客选路失败
    TNKDriveRouteSearchErrorCode_Privacy_Failed               = 2009,     ///< 尚未同意隐私协议
    TNKDriveRouteSearchErrorCode_ServerError                  = 2999      ///< 服务器内部错误
} TNKDriveRouteSearchErrorCode;                           ///< 路线规划的错误码

typedef enum _TNKRoadType
{
    TNKRoadType_None                   = 0,        ///< 无详细类型
    TNKRoadType_Elevated               = 1,        ///< 在桥上
    TNKRoadType_Downstairs             = 2,        ///< 在桥下
    TNKRoadType_MainRoad               = 3,        ///< 在主路
    TNKRoadType_ServingRoad            = 4,        ///< 在辅路
    TNKRoadType_DirectionRoad          = 5,        ///< 在对面，初始算路无效
    TNKRoadType_DownstairsMainRoad     = 6,        ///< 在桥下主路
    TNKRoadType_DownstairsServingRoad  = 7,        ///< 在桥下辅路
} TNKRoadType;         ///< 路段的详细类型

typedef enum _TNKCameraType
{
    TNKCameraTypeNone                 = 0,  ///< 无类型
    TNKCameraTypeRedLight             = 1,  ///< 闯红灯照相
    TNKCameraTypeElectronicMonitoring = 2,  ///< 电子监控
    TNKCameraTypeFixedSpeedTraps      = 3,  ///< 固定测速点
    TNKCameraTypeMobileSpeedZone      = 4,  ///< 移动测速区
    TNKCameraTypeBusOnlyWay           = 5,  ///< 公交
    TNKCameraTypeOneWay               = 6,  ///< 单行
    TNKCameraTypeEmergencyWay         = 7,  ///< 应急
    TNKCameraTypeNoneMotorWay         = 8,  ///< 非机动车
    TNKCameraTypeQujianEnter          = 9,  ///< 区间测速进入点
    TNKCameraTypeQujianExit           = 10, ///< 区间测速退出点
    TNKCameraTypeHOV                  = 16, ///< 多乘员专用道监控.
    TNKCameraTypeLaLian               = 17, ///< 拉链通行道路监控.
    TNKCameraTypeTailNumber           = 21, ///< 尾号限行，
    TNKCameraTypeGoToBeijing          = 22, ///< 外地车进京监控,
    TNKCameraTypeIllegalBlow          = 23, ///< 违法鸣笛摄像头
    TNKCameraTypeBusStation           = 24, ///< 公交车站摄像头
    TNKCameraTypeForbiddenTure        = 30, ///< 禁止左右转
    TNKCameraTypeForbiddenLine        = 31, ///< 违反禁止标线
    TNKCameraTypeForbiddenParking     = 32, ///< 违章停车
    TNKCameraTypeLowestSpeed          = 33, ///< 超低速
    TNKCameraTypeAlterableSpeed       = 34, ///< 可变限速
    TNKCameraTypeLaneSpeed            = 35, ///< 分车道限速
    TNKCameraTypeVehicelTypeSpeed     = 36, ///< 分车种限速
    TNKCameraTypeLaneOccupy           = 37, ///< 违章占车道
    TNKCameraTypeCrossing             = 38, ///< 违章过路口
    TNKCameraTypeForbiddenSign        = 39, ///< 违反禁令标志
    TNKCameraTypeForbiddenLight       = 40, ///< 违规用灯
    TNKCameraTypeLifeBelt             = 41, ///< 不系安全带
    TNKCameraTypeForbiddenCall        = 42, ///< 开车打手机
    TNKCameraTypeLimitLine            = 43, ///< 环保限行
    TNKCameraTypePedestrainFirst      = 44, ///< 礼让行人
    TNKCameraTypeAnnualInpection      = 45, ///< 车辆未按规定年检
    TNKCameraTypeVehicelExhaust       = 46, ///< 车辆尾气超标
    TNKCameraTypeTraffic              = 47, ///< 路况监控
    TNKCameraTypeEntrance             = 48, ///< 出入口摄像头
    TNKCameraTypeForbiddenUTurn       = 49, ///< 禁止掉头摄像头
    TNKCameraTypeEtcToll              = 50, ///< etc收费电子眼
    TNKCameraTypeNotFollowGuideLane   = 51, ///< 不按导向车道行驶
    TNKCameraTypeTrafficFlowMonitor   = 52, ///< 流量监控（车流量）
    TNKCameraTypeKeepSafeDistance     = 53, ///< 保持安全距离
    TNKCameraTypeIllegalChangeLane    = 54  ///< 违法变道
} TNKCameraType; ///< 电子眼的类型


typedef enum _TNKTurnRestrictionType
{
    TNKTurnRestrictionTypeNone = 0,         ///< 无禁止
    TNKTurnRestrictionTypeLeft = 1,         ///< 禁止左转
    TNKTurnRestrictionTypeRight = 2,        ///< 禁止右转
    TNKTurnRestrictionTypeLeftUTurn = 3,    ///< 禁止左调头
    TNKTurnRestrictionTypeRightUTurn = 4,   ///< 禁止右调头
    TNKTurnRestrictionTypeStraight = 5,     ///< 禁止直行
} TNKTurnRestrictionType; ///< 转向禁止类型


typedef enum _TNKTrafficLightType
{
    TNKTrafficLightTypeNone = -1,   ///< 无效灯态
    TNKTrafficLightTypeRed = 0,     ///< 红灯
    TNKTrafficLightTypeGreen = 1,   ///< 绿灯
    TNKTrafficLightTypeYellow = 2,  ///< 黄灯
} TNKTrafficLightType; ///< 红绿灯灯态

/// 驾车限行类型
typedef NS_OPTIONS(NSUInteger, TNKDriveLimitStatus) {
    TNKDriveLimitStatusNone             = 0,            ///< 没有经过限行区域
    TNKDriveLimitStatusAvoided          = 1 << 0,       ///< 避开限行区域
    TNKDriveLimitStatusLimitAtStart     = 1 << 1,       ///< 起点在限行区域
    TNKDriveLimitStatusLimitAtEnd       = 1 << 2,       ///< 终点在限行区域
    TNKDriveLimitStatusThrough          = 1 << 3,       ///< 经过限行路段
    TNKDriveLimitStatusCanAvoid         = 1 << 4,       ///< 可以绕过限行区域（在未开启避开限行选项时设置）
};

typedef enum _TNKSliceType
{
    TNKSliceTypeNone    = 0,        ///< 不分片
    TNKSliceTypeTwo     = 1,        ///< 两段
    TNKSliceTypeThree   = 2,        ///< 三段
    TNKSliceTypeAuto    = 9,        ///< 自动
} TNKSliceType;         ///< 分包分片的详细类型


typedef enum _TNKEnergyType
{
    TNKEnergyTypeUnknown = 0,       ///< 未知
    TNKEnergyTypeDiesel,            ///< 柴油
    TNKEnergyTypeHybird,            ///< 油电混动
    TNKEnergyTypeElectric,          ///< 纯电
    TNKEnergyTypeGas,               ///< 汽油
} TNKEnergyType; ///< 能源类型

/*------------------------------------导航过程------------------------------------*/

/**
 * @brief 导航开启失败信息
 */
@interface TNKNavStartFailInfo : NSObject

/**
 * @brief 导航开启失败code
 */
@property (nonatomic, assign) int code;

/**
 * @brief 导航开启失败信息
 */
@property (nonatomic, copy) NSString *msg;

/**
 * @brief 路线ID
 */
@property (nonatomic, copy) NSString *routeID;

@end
typedef enum _TNKRoadMatchStatus
{
    TNKRoadMatchStatusNull = -1,            ///< 默认值
    TNKRoadMatchStatusOffRoute = 0,         ///< 车标脱离
    TNKRoadMatchStatusOnRoute = 1,          ///< 车标匹配在路上
} TNKRoadMatchStatus;   ///< 道路吸附结果



/**
 * @brief 驾车路径规划算路策略
 */
@interface TNKDriveRouteSearchOption : NSObject<NSCopying>

/**
 * @brief 驾车路线规划参数: 是否避开收费道路, 默认NO, 即不避开收费道路.
 */
@property (nonatomic, assign) BOOL avoidToll;

/**
 * @brief 驾车路线规划参数: 是否避开高速道路, 默认为NO, 即不避开高速道路.
 */
@property (nonatomic, assign) BOOL avoidHighway;

/**
 * @brief 驾车路线规划参数: 是否结合路况, 默认值NO, 即不结合路况.
 */
@property (nonatomic, assign) BOOL avoidTrafficJam;

/**
 * @brief 驾车路线规划参数: 是否优先高速, 默认NO. since 5.3.3
 */
@property (nonatomic, assign) BOOL highway;

/**
 * @brief 驾车路线规划参数: 是否优先出时间短的路线, 默认为NO. since 5.3.3
 */
@property (nonatomic, assign) BOOL shortTime;

/**
 * @brief 驾车路线规划参数: 是否优先出距离短的路线, 默认为false. since 6.11.0
 */
@property (nonatomic, assign) BOOL isShortDistance;

/**
 * @brief 驾车路线规划参数: 是否优先大路, 默认值NO. since 5.3.3
 */
@property (nonatomic, assign) BOOL bigRoad;

/**
 * @brief 起点路段类型：设置详细的起点路段类型, 如在桥下、在辅路等. 设置该字段可以提升路线规划起点位置的准确程度. 默认为无详细起点路段类型.
 */
@property (nonatomic, assign) TNKRoadType startRoadType;
    
/**
 * @brief 车牌号：设置车牌号后可接收限行信息, 可以为空.
 */
@property (nonatomic, copy, nullable) NSString *carNumber;

/**
 * @brief 是否为新能源车。
 * @since 5.4.1
 */
@property (nonatomic, assign) BOOL greenCar;

/**
 * @brief 驾车能源类型
 */
@property (nonatomic, assign) TNKEnergyType carEnergyType;

/**
 * @brief 路径规划返回多方案。默认为YES
 * @since 6.0.0
 */
@property (nonatomic, assign) BOOL searchMultiRoutes;

/**
 * @brief 导航前序点，提升路线规划的准确性。前序点整体是非必传字段，可以提高起点准确性，前序点里的字段 经纬度，角度，精度，速度，时间戳 为必传. since 5.3.9更改为TencentLBSLocation数组
 */
@property (nonatomic, copy, nullable) NSArray<TencentLBSLocation *> *preLocations;

/**
 * @brief 导航点串还原导航路线 since 6.13.0
 */
@property (nonatomic, copy, nullable) NSArray<TencentLBSLocation *> *routePointsLocations;

/**
 * @brief 导航点串还原导航路线，行前算路设置。默认为NO
 * @since 6.13.0
 */
@property (nonatomic, assign) BOOL routePointsIsSearch;

/**
 * @brief 导航点串还原导航路线，行中重算。默认为YES
 * @since 6.13.0
 */
@property (nonatomic, assign) BOOL routePointsIsReRouteSearch;

/**
 * @brief 是否规避限行。默认为YES
 * @since 6.2.0
 */
@property (nonatomic, assign) BOOL avoidLimit;

/**
 * @brief 分包分片策略
 * @since 6.9.0
 */
@property (nonatomic, assign) TNKSliceType sliceType;


/*------------------------------------货车导航参数------------------------------------*/

/**
 * @brief 货车算路属性. since 5.2.1
 */
@property (nonatomic, copy, nullable) TNKTruckSearchOption *truckSearchOption;


/*------------------------------------出行场景参数------------------------------------*/

/**
 * @brief 驾车路线规划参数: 导航场景. 1表示去接乘客, 2表示去送乘客. 默认值为0.
 */
@property (nonatomic, assign) int navScene;

/**
 * @brief 客户端的用户id
 */
@property (nonatomic, copy, nullable) NSString *userID;

/**
 * @brief 行前乘客选路的路线id，可在webservice api的路径规划接口中获得. since 5.3.3
 */
@property (nonatomic, copy, nullable) NSString *initialRouteID;

/**
 * @brief 行前乘客选路的request_id，可在webservice api的路径规划接口中获得. since 5.3.3
 */
@property (nonatomic, copy, nullable) NSString *initialRequestID;

@end

/**
 * @brief 驾车算路请求类
 */
@interface TNKDriveRouteRequest: TNKRouteRequest

/**
 * @brief 途经点数组，最多支持32个。
 */
@property (nonatomic, copy, nullable) NSArray<TNKSearchNavPoint *> *waypoints;

/**
 * @brief 驾车路径规划算路策略
 */
@property (nonatomic, copy) TNKDriveRouteSearchOption *searchOption;

/**
 * @brief 路线规划避让区域。最多支持32个避让区域，每个区域最多可设置16个顶点，需为凸多边形，顶点逆时针排列。
 * 注意：如果起终点在避让区域内，则算路失败，错误码TNKDriveRouteSearchErrorCode_CalculateFailed
 * 如果数据不合法，则避让策略不生效。since 5.2.5
 *
 *  @return 错误码。0: 正确;1: 避让区域超过32个；2: 设置区域的顶点超过9个
 */
- (int)setAvoidAreas:(NSArray<NSArray<TNKCoordinatePoint *> *> *)avoidAreas;

@end

/**
 * @brief 算路的路线数据类
 */
@interface TNKDriveRoute : NSObject

/**
 * @brief 路线标签。比如：时间段、距离短等。
 */
@property (nonatomic, readonly) NSString *recommendReason;

/**
 * @brief 组成路线形状的坐标点数组
 */
@property (nonatomic, readonly) NSArray<TNKCoordinatePoint *> *routePoints;

/**
 * @brief 路线ID
 */
@property (nonatomic, readonly) NSString *routeID;

/**
 * @brief 总距离. 单位，米
 */
@property (nonatomic, readonly) int totalDistance;

/**
 * @brief 总预计时间. 单位，分钟
 */
@property (nonatomic, readonly) int totalTime;

/**
 * @brief 路线红绿灯个数
 */
@property (nonatomic, readonly) int trafficLightCount;

/**
 * @brief 路线预计所需高速费 单位: 元
 */
@property (nonatomic, readonly) int fee;

/**
 * @brief 路线预计收费路段的里程，单位:米
 * @since 6.3.0
 */
@property (nonatomic, readonly) int tollDistance;

/**
 * @brief 导航起点
 */
@property (nonatomic, readonly) TNKSearchNavPoint *origin;

/**
 * @brief 导航终点
 */
@property (nonatomic, readonly) TNKSearchNavPoint *destination;

/**
 * @brief 途经点信息, 可为空
 */
@property (nonatomic, readonly, nullable) NSArray<TNKNavToWayPointInfo *> *waypoints;

/**
 * @brief 路线规划时的路况状态数据.
 */
@property (nonatomic, readonly) NSArray <TNKRouteTrafficItem *> *routeTrafficItems;

/**
 * @brief 路线规划时的路线segment数据.
 */
@property (nonatomic, readonly) NSArray <TNKSegmentItem *> *segmentItems;

/**
 * @brief 路线解释性信息. since 5.2.4
 */
@property (nonatomic, readonly, nullable) NSArray <TNKDriveRouteExplain *> *routeExplainItems;

/**
 * @brief 路线红绿灯信息.
 * @since 5.3.1
 */
@property (nonatomic, readonly, nullable) NSArray <TNKRoutePoint *> *trafficLights;

/**
 * @brief 与其他路线的分歧点信息
 */
@property (nonatomic, readonly, nullable) NSArray<TNKForkPoint *> *forkPoints;

/**
 * @brief 备选路线推路线推荐的理由。-1-默认，0-更快，1-封路，2-限行。
 * @since 5.4.1
 */
@property (nonatomic, readonly) int bkRecommendType;

/**
 * @brief 违反交规信息类列表
 * @since 6.3.0
 */
@property (nonatomic, readonly) NSArray<TNKTurnRestrictionItem *> *turnRestrictionItems;

/**
 * @brief 违反货车限制交规信息列表
 * @since 6.3.0
 */
@property (nonatomic, readonly) NSArray<TNKTruckRestrictionItem *> *truckRestrictionItems;

/**
 * @brief 个性化路线客户标签
 * @since 6.4.0
 */
@property (nonatomic, readonly, nullable) NSArray<NSNumber *> *customTags;

/**
 * @brief 限行信息。 通过(limitStatus  & TNKDriveLimitStatusLimitAtStart)、(limitStatus  & TNKDriveLimitStatusLimitEnd)去判断限行状态
 * @since 6.5.0
 */
@property (nonatomic, readonly) TNKDriveLimitStatus limitStatus;

/**
 * @brief 货车经验路线id。没有则为空字符串
 * @since 6.6.0
 */
@property (nonatomic, readonly) NSString *experienceRouteID;

@end

/**
 * @brief 驾车算路返回数据类
 */
@interface TNKDriveRouteSearchResult : NSObject

/**
 * @brief 路线数组，最多三个
 */
@property (nonatomic, readonly) NSArray<TNKDriveRoute *> *routes;

/**
 * @brief 路线还原是否成功。如果成功，routes的第一个路线为还原路线
 */
@property (nonatomic, readonly) BOOL isSearchByRouteIDWorking;

/**
 * @brief 是否命中终点延伸策略. @since 5.3.9
 */
@property (nonatomic, readonly) BOOL hitEndAdsorbExtStrategy;

/**
 * @brief 导航的会话id.
 * @since 5.4.1
 */
@property (nonatomic, readonly) NSString *naviSessionID;


/// 个性化路线的解释信息
@property (nonatomic, readonly, nullable) TNKCustomRouteReason *customRouteReason;

@end


typedef enum _TNKTruckType
{
    TNKTruckTypeMiniTruck                 = 1,                  ///< 微型车
    TNKTruckTypeLightTruck                = 2,                  ///< 轻型车 （默认）
    TNKTruckTypeMediumTruck               = 3,                  ///< 中型车
    TNKTruckTypeHeavyTruck                = 4,                  ///< 重型车
} TNKTruckType;     ///< 货车类型


typedef enum _TNKTruckTrailerType
{
    TNKTruckTrailerTypeUnknown = 0, ///< 未知
    TNKTruckTrailerTypeFlatbed,     ///< 平板
    TNKTruckTrailerTypeContainer,   ///< 箱货
    TNKTruckTrailerTypeSemi,        ///< 半挂
    TNKTruckTrailerTypeFull,        ///< 全挂
} TNKTruckTrailerType; ///< 拖挂类型

typedef enum _TNKTruckPlateColor
{
    TNKTruckPlateColorUnknown = 0,  ///< 未知
    TNKTruckPlateColorBlue,         ///< 蓝色
    TNKTruckPlateColorYellow,       ///< 黄色
    TNKTruckPlateColorBlack,        ///< 黑色
    TNKTruckPlateColorWhite,        ///< 白色
    TNKTruckPlateColorGreen,        ///< 绿色
    TNKTruckPlateColorYellowGreen,  ///< 黄绿
} TNKTruckPlateColor; ///< 车牌颜色


typedef enum _TNKTruckFunctionType
{
    TNKTruckFunctionTypeNormal = 0,         ///< 常规
    TNKTruckFunctionTypeDanger,             ///< 危险品
} TNKTruckFunctionType; ///< 货车用途

typedef enum _TNKEmissionStandard
{
    TNKEmissionStandardUnknown = 0,         ///< 未知
    TNKEmissionStandardI,                   ///< 国一
    TNKEmissionStandardII,                  ///< 国二
    TNKEmissionStandardIII,                 ///< 国三
    TNKEmissionStandardIV,                  ///< 国四
    TNKEmissionStandardV,                   ///< 国五
    TNKEmissionStandardVI,                  ///< 国六
} TNKEmissionStandard; ///< 排放标准

typedef enum _TNKTruckPassType
{
    TNKTruckPassTypeNoNeed,             ///< 不需要
    TNKTruckPassTypeHave,               ///< 有通行证
    TNKTruckPassTypeNotHave,            ///< 无通行证
    TNKTruckPassTypeApproval,           ///< 预约通行证
} TNKTruckPassType; ///< 通行证类型


typedef enum _TNKTruckRestrictionType
{
    TNKTruckRestrictionTypeNone   = 0,     ///< 无限制
    TNKTruckRestrictionTypeWidth  = 1,     ///< 限宽
    TNKTruckRestrictionTypeHeight = 2,     ///< 限高
    TNKTruckRestrictionTypeLength = 3,     ///< 限长
    TNKTruckRestrictionTypeWeight = 4,     ///< 限重
    TNKTruckRestrictionTypeDanger = 5,     ///< 限制危险品
} TNKTruckRestrictionType;     ///< 货车限制类型

/**
 * @brief 货车导航算路参数.
 * @since 5.2.1
 */
@interface TNKTruckSearchOption : NSObject<NSCopying>

/**
 * @brief 货车类型， 默认为轻型车
 */
@property (nonatomic, assign) TNKTruckType truckType;

/**
 * @brief 长度，单位：m
 */
@property (nonatomic, assign) float length;

/**
 * @brief 宽度，单位：m
 */
@property (nonatomic, assign) float width;

/**
 * @brief 高度，单位：m
 */
@property (nonatomic, assign) float height;

/**
 * @brief 总重量，单位：吨
 */
@property (nonatomic, assign) float weight;

/**
 * @brief 核定载重,单位：吨
 */
@property (nonatomic, assign) float load;

/**
 * @brief 轴数
 */
@property (nonatomic, assign) int axisCount;

/**
 * @brief 轴重,单位：吨
 */
@property (nonatomic, assign) float axisLoad;

/**
 * @brief 拖车类型
 */
@property (nonatomic, assign) TNKTruckTrailerType trailerType;

/**
 * @brief 车牌颜色。
 */
@property (nonatomic, assign) TNKTruckPlateColor plateColor;

/**
 * @brief 能源类型
 */
@property (nonatomic, assign) TNKEnergyType energyType;

/**
 * @brief 功能类型
 */
@property (nonatomic, assign) TNKTruckFunctionType functionType;

/**
 * @brief 尾气排放标准
 */
@property (nonatomic, assign) TNKEmissionStandard emissionStandard;

/**
 * @brief 通行证类型
 */
@property (nonatomic, assign) TNKTruckPassType passType;

/**
 * @brief 算路时是否忽略重量。默认为NO
 */
@property (nonatomic, assign) BOOL willIgnoreWeight;


/**
 * @brief 算路时是否需要请求经验路线。默认为NO
 * @since 6.6.0
 */
@property (nonatomic, assign) BOOL needExperienceRoute;

@end

/**
 * @brief 多路线分歧点信息.
 * @since 5.2.6
 */
@interface TNKForkPoint : NSObject

/**
 * @brief 分歧点坐标
 */
@property (nonatomic, readonly) CLLocationCoordinate2D coord;
/**
 * @brief 分歧点所在索引
 */
@property (nonatomic, readonly) int pointIndex;
/**
 * @brief 与之分歧的路线id
 */
@property (nonatomic, readonly) NSString *forkRouteID;

@end


/**
 * @brief 导航过程中的路线实时数据
 */
@interface TNKDriveNavRoute : NSObject

- (instancetype)init NS_UNAVAILABLE;

/// 构造方法
- (instancetype)initWithSearchRoute:(TNKDriveRoute *)searchRoute;

/**
 * @brief 算路的路线数据类
 */
@property (nonatomic, readonly) TNKDriveRoute *searchRoute;

/**
 * @brief 当前位置到终点剩余距离， 单位：米
 */
@property (nonatomic, readonly) int remainingDistance;

/**
 * @brief 当前位置到终点剩余时间，单位：分钟
 */
@property (nonatomic, readonly) int remainingTime;

/**
 * @brief 当前位置到终点剩余时间，单位：秒
 */
@property (nonatomic, readonly) int remainingTimeInSeconds;

/**
 * @brief 当前路况数据
 */
@property (nonatomic, readonly) NSArray<TNKRouteTrafficItem *> *curTrafficItems;

/**
 * @brief 当前定位信息
 */
@property (nonatomic, readonly, nullable) TNKLocation *location;

/**
 * @brief 导航的起点与当前路线的首个点的距离。如果这条路线是偏航重算的路线，那么该值就是从导航开始到偏航时走过的距离。
 */
@property (nonatomic, readonly) int distBetweenOrginAndRouteFistPoint;

@end

/**
 * @brief 导航中前方电子眼信息
 */
@interface TNKNavCameraInfo : NSObject

/**
 * @brief 电子眼类型
 */
@property (nonatomic, assign) TNKCameraType type;

/**
 * @brief 电子眼位置信息
 */
@property (nonatomic, strong) TNKRoutePoint *routePoint;

/**
 * @brief 距电子眼的剩余距离，单位：米
 */
@property (nonatomic, assign) int distance;

/**
 * @brief 限速信息，单位为km/h。如果是限速电子眼，则该字段数据有效
 */
@property (nonatomic, assign) int limitSpeed;

/**
 * @brief 区间限速长度，单位为米。如果是区间限速起点电子眼，则该字段数据有效
 */
@property (nonatomic, assign) int speedZoneLimitLength;

@end

/**
 * @brief 交规信息基础类
 */
@interface TNKRestrictionItem : NSObject

/**
 * @brief 位置信息
 */
@property (nonatomic, readonly) TNKRoutePoint *routePoint;

/**
 * @brief 所在道路名
 */
@property (nonatomic, readonly) NSString *roadName;

@end

/**
 * @brief 违反货车规格限制信息
 */
@interface TNKTruckRestrictionItem : TNKRestrictionItem

/**
 * @brief 限制类型
 */
@property (nonatomic, readonly) TNKTruckRestrictionType type;

/**
 * @brief 当type为TruckRestrictionTypeWidth是，有合法值
 */
@property (nonatomic, readonly) float limitWidth;

/**
 * @brief 当type为TruckRestrictionTypeHeight是，有合法值
 */
@property (nonatomic, readonly) float limitHeight;

/**
 * @brief 当type为TruckRestrictionTypeLength是，有合法值
 */
@property (nonatomic, readonly) float limitLength;

/**
 * @brief 当type为TruckRestrictionTypeWeight是，有合法值
 */
@property (nonatomic, readonly) float limitWeight;

@end

/**
 * @brief 违反交规信息
 */
@interface TNKTurnRestrictionItem : TNKRestrictionItem

/**
 * @brief 禁止类型
 */
@property (nonatomic, readonly) TNKTurnRestrictionType type;

/**
 * @brief 是否全天禁止
 */
@property (nonatomic, readonly) BOOL always;

/**
 * @brief 非全天限制时的时间段限制信息 格式 %HH:%MM~%HH:%MM 多组以;号分隔
 */
@property (nonatomic, readonly) NSString *timeDesc;

/**
 * @brief int的低5位从右到左分别标识，微型货车、轻型货车 中型货车 重型货车 小客车 是否限制
 */
@property (nonatomic, readonly) int limitCarType;

@end

/**
 * @brief 实时交规信息基础类
 */
@interface TNKNavRestrictionItem : NSObject

/**
 * @brief 交规信息基础类
 */
@property (nonatomic, strong, readonly) TNKRestrictionItem *restrictionItem;

/**
 * @brief 当前位置到达该点的剩余距离，单位：米
 */
@property (nonatomic, readonly) int distance;

@end

/**
 * @brief 实时违反货车规格限制信息
 */
@interface TNKTruckNavRestrictionItem : TNKNavRestrictionItem

/**
 * @brief 违反货车规格限制信息
 */
@property (nonatomic, strong, readonly) TNKTruckRestrictionItem *truckRestrictionItem;

@end

/**
 * @brief 实时违反交规信息
 */
@interface TNKTurnNavRestrictionItem : TNKNavRestrictionItem

/**
 * @brief 违反交规信息
 */
@property (nonatomic, strong, readonly) TNKTurnRestrictionItem *turnRestrictionItem;

@end

/**
 * @brief 导航中前方交规信息
 */
@interface TNKNavRestrictionInfo : NSObject

/**
 * @brief 导航中前方最近的货车限制信息
 */
@property (nonatomic, readonly) NSArray<TNKTruckNavRestrictionItem *> *truckNavRestrictionItems;

/**
 * @brief 导航中前方最近的交规信息
 */
@property (nonatomic, readonly) NSArray<TNKTurnNavRestrictionItem *> *turnNavRestrictionItems;

@end


/**
 * @brief 被过滤的Link
 */
@interface TNKRouteFilterLink : NSObject

/// link id
@property (nonatomic, readonly) NSString *linkID;

/// 过滤原因
@property (nonatomic, readonly) int  filterType;

/// 坐标
@property (nonatomic, readonly) CLLocationCoordinate2D coord;

@end


@interface TNKCustomRouteReason : NSObject

/**
 * @brief 被过滤的Link数组
 */
@property (nonatomic, readonly) NSArray<TNKRouteFilterLink *> *filterLinks;

/**
 * @brief 经验路线状态。0=没有指定路线 1=指定路线引擎未召回 2=指定路线被过滤 3=指定路线成功返回
 */
@property (nonatomic, readonly) int customerState;

@end

/// 红绿灯倒计时信息
@interface TNKTrafficLightCountdownMsg : NSObject

/// 红绿灯的位置信息
@property (nonatomic, readonly) TNKRoutePoint *routePoint;
/// 当前的灯态，为TNKTrafficLightTypeNone时表示没有红绿灯
@property (nonatomic, readonly) TNKTrafficLightType trafficLightType;
/// 灯态结束的时间戳，单位:秒
@property (nonatomic, readonly) int endTimestamp;
/// 下一个灯态，为TNKTrafficLightTypeNone时表示没有红绿灯
@property (nonatomic, readonly) TNKTrafficLightType nextTrafficLightType;
/// 下一个灯态结束的时间戳，单位:秒
@property (nonatomic, readonly) int nextEndTimestamp;

@end


/**
 * @brief 巡航道路吸附信息
 */
@interface TNKNavRoadMatchResult : NSObject

/**
 * @brief 巡航道路吸附状态
 */
@property (nonatomic, readonly) TNKRoadMatchStatus roadMatchStatus;

/**
 * @brief 吸附坐标
 */
@property (nonatomic, readonly) CLLocation *mactchLocation;

/**
 * @brief 道路等级
 */
@property (nonatomic, readonly) TNKRoadGrade roadGrade;

/**
 * @brief 道路属性
 */
@property (nonatomic, readonly) TNKRoadKind roadKind;

/**
 * @brief 道路方向[0,360)
 */
@property (nonatomic, readonly) double roadDir;

@end

/**
 * @brief 巡航定位信息
 */
@interface TNKPosMatchLocation: NSObject

/**
 * @brief 原始定位
 */
@property (nonatomic, readonly) TencentLBSLocation *origLocation;

/**
 * @brief 巡航道路吸附信息
 */
@property (nonatomic, readonly) TNKNavRoadMatchResult *roadMatchResult;


@end


NS_ASSUME_NONNULL_END
