//
//  TNKBaseNavView.h
//  TencentNavKit
//
//  Created by mol on 2022/10/20.
//  Copyright © 2022 Tencent. All rights reserved.
//

#import <UIKit/UIKit.h>
#import <TencentNavKit/TNKNavViewCommonObj.h>
#import <TencentNavKit/TNKRouteConfig.h>
#import <TencentNavKit/TNKODMarkerPresentation.h>
#import <TencentNavKit/TNKUserLocationPresentation.h>
#import <TencentNavKit/TNKCommonObj.h>
#import <QMapKit/QMapView.h>
#import <QMapKit/QMapView+Config.h>
#import <TencentNavKit/TNKRedLineConfig.h>
#import <TencentNavKit/TNKTurnArrowConfig.h>


NS_ASSUME_NONNULL_BEGIN


/// 导航地图基础类
@interface TNKBaseNavView : UIView

- (instancetype)init UNAVAILABLE_ATTRIBUTE;

/**
 *  @brief 构造方法 . since 5.2.4.4
 *  @param frame    frame 矩形.
 *  @param config 地图 的配置属性.
 */
- (instancetype)initWithFrame:(CGRect)frame config:(QMapConfig * _Nullable)config;

/**
 * @brief 地图实例
 */
@property (nonatomic, readonly) QMapView *mapView;

/**
 *  @brief 地图日夜间模式切换策略. 包括自动切换、始终白天、始终黑夜3种模式. 默认为日夜状态自动切换.
 */
@property (nonatomic, assign) TNKNavDayNightMode dayNightMode;

/**
 *  @brief 获取地图当前日夜状态.
 */
@property (nonatomic, readonly) TNKNavDayNightStatus dayNightStatus;

/**
 *  @brief 导航视角模式. 导航模式包括3D车头朝上、2D地图朝北、全览和剩余全览4种模式. 默认为3D车头朝上模式.
 *         导航状态下使用手势操作地图时触发非锁车模式，地图不跟随自车点移动 ；在手势操作结束后一段时间切换回之前的导航模式.
 *         该时间可由followDelay设定,默认回弹时间为5秒。
 */
@property (nonatomic, assign) TNKNavUIMode navUIMode;

/**
 * @brief 当前是否是锁车标状态。当收到拖动地图时，地图视野一段时间内不会发生改变，follow为NO。
 */
@property (nonatomic, assign) BOOL followed;

/**
 * @brief 是否会自动从非锁车标状态进入锁车标状态； 默认为YES
 */
@property (nonatomic, assign) BOOL followEnabled;

/**
 *  @brief 设置从非锁车标状态返回之前锁车标状态的延迟时间，单位为秒， 默认为5秒.
 */
@property (nonatomic, assign) NSTimeInterval followDelay;

/**
 * @brief 是否会自动从全览状态进入导航态,默认为NO
 */
@property (nonatomic, assign) BOOL exitOverviewEnabled;

/**
 *  @brief 设置从全览到退出全览模式进入导航态的延迟时间,单位为秒,默认为10秒,区间(0,+∞).
 */
@property (nonatomic, assign) NSTimeInterval exitOverviewDelay;

/**
 *  @brief 非全览的导航视角模式. 导航模式包括3D车头朝上、2D地图朝.
 */
@property (nonatomic, assign) TNKNavUIMode nonOverviewNaviUIMode;

/**
 *  @brief 全览的导航视角模式. 全览和剩余全览.
 */
@property (nonatomic, assign) TNKNavUIMode overviewNaviUIMode;


/**
 * @brief 驾车导航路线样式配置
 * @note TNKCarRouteColorPalette 中的eraseColor 不支持动态更新.
 */
@property (nonatomic, readonly) TNKRouteConfig *routeConfig;


/**
 *  @brief 清除路线相关UI.该方法用于非导航状态下清除路线相关UI。
 *         包括导航路线、自车点图标、动态添加的限速/指示标识等.
 *   @warning 导航过程中调用该方法无效.
 */
- (void)clearAllRouteUI;

/**
 *  @brief 设置全览和剩余全览模式下导航路线显示区域距离地图四周的边距.
 *          当在CarNaviView上自定义显示元素时, 为避免与导航路线相互遮挡, 可调用该接口调整导航路线显示区域范围.
 *          默认值为(80,80,150,40), 使得全览和剩余全览模式下导航路线四周有部分留白.
 */
@property (nonatomic, assign) UIEdgeInsets externalEdgeInsets;

/**
 *  @brief 设置全览和剩余全览模式下，想要展示在可视区域内的额外的点数组。使用之后需重置为nil
 */
@property (nonatomic, copy, nullable) NSArray<TNKCoordinatePoint *> *extraPointsForOverview;

/**
 *  @brief 3D导航时自车点位置在地图上位置的比例 默认值为(0.5,0.72).
 *  x取值范围为[0.25,0.75] 表示从屏幕左边起的宽度比例.
 *  y取值范围为[0.25,0.75] 表示从屏幕上边起的高度比例.
 *  若设置值不在区间内则使用默认值.
 */
@property (nonatomic, assign) CGPoint fix3DProportion;

/**
 *  @brief 2D导航时自车点位置在地图上位置的比例 默认值为(0.5,0.6).
 *  x取值范围为[0.25,0.75] 表示从屏幕左边起的宽度比例.
 *  y取值范围为[0.25,0.75] 表示从屏幕上边起的高度比例.
 *  若设置值不在区间内则使用默认值.
 */
@property (nonatomic, assign) CGPoint fix2DProportion;

/**
 *  @brief 设置是否隐藏自车点罗盘.该方法用于导航状态下控制显示/隐藏自车点罗盘. 
 *          自车点罗盘在3D车头朝上模式下会默认显示, 在2D地图朝北、全览模式和剩余全览模式下会默认隐藏.
 */
@property (nonatomic, assign) BOOL compassHidden;


/**
 *  @brief 设置是否隐藏转弯箭头指引.该方法用于导航状态下控制显示/隐藏下一个转弯箭头指引.默认为YES
 */
@property (nonatomic, assign) BOOL turnArrowEnabled;

/**
 *  @brief 设置是转向箭头的样式，不可为nil
 */
@property (nonatomic, strong) TNKTurnArrowConfig *turnArrowConfig;

/**
 *  @brief 更新路线样式配置
 *  @param routeConfig 路线样式配置
 */
- (void)updateRouteConfig:(TNKRouteConfig *)routeConfig;

/**
 *  @brief 更新起点样式。 置为nil则重新使用默认资源
 *  @param originMarkerPresentation 起点样式
 */
- (void)updateOriginMarkerPresentation:(TNKODMarkerPresentation * _Nullable)originMarkerPresentation;

/**
 *  @brief 更新终样式。 置为nil则重新使用默认资源
 *  @param destinationMarkerPresentation 终点样式
 */
- (void)updateDestinationMarkerPresentation:(TNKODMarkerPresentation * _Nullable)destinationMarkerPresentation;

/**
 *  @brief 更新自车点样式。 置为nil则重新使用默认资源
 *  @param userLocationPresentation 自车点样式
 */
- (void)updateUserLocationPresentation:(TNKUserLocationPresentation * _Nullable)userLocationPresentation;

/**
 * @brief 配置导航红线。默认不显示，入参为nil则不显示
 * @since 6.0.0
 */
- (void)updateRedLineConfig:(TNKRedLineConfig * _Nullable)redLineConfig;

@end

@interface TNKBaseNavView (QMapView)


/**
 * @brief 开启定位并展示位置图标.
 */
@property (nonatomic, assign) BOOL showsUserLocation;

/**
 * @brief 配置当前位置UI属性 (必须在QMapView 初始化以后, 定位开启之前调用)
 *
 * @param presentation 位置UI属性
 */
- (void)configureUserLocationPresentation:(QUserLocationPresentation *)presentation;

/**
 *  @brief 改变底图样式。注意，如果调用该方法，日夜间模式样式将不再发生变化.
 *  @since 5.2.4.4
 *  @param styleType 在官网配置的自定义样式列表中的顺序, 从1开始
 */
- (void)setMapStyle:(int)styleType;

@end

NS_ASSUME_NONNULL_END
