#import <Foundation/Foundation.h>
#import "QMapResultSet.h"
#import "QMapDatabasePool.h"

NS_ASSUME_NONNULL_BEGIN

// 编译环境是否启用了ARC
#if ! __has_feature(objc_arc)
    #define QMapDBAutorelease(__v) ([__v autorelease]);
    #define QMapDBReturnAutoreleased QMapDBAutorelease

    #define QMapDBRetain(__v) ([__v retain]);
    #define QMapDBReturnRetained QMapDBRetain
    #define QMapDBRelease(__v) ([__v release]);

    #define QMapDBDispatchQueueRelease(__v) (dispatch_release(__v));
#else
    // -fobjc-arc
    #define QMapDBAutorelease(__v)
    #define QMapDBReturnAutoreleased(__v) (__v)

    #define QMapDBRetain(__v)
    #define QMapDBReturnRetained(__v) (__v)
    #define QMapDBRelease(__v)

// 如果 OS_OBJECT_USE_OBJC=1，则调度对象将被视为 ObjC 对象, 并将启用ARC，涉及到(GCD) 参考”中的“调度队列和自动引用计数”部分。
    #if OS_OBJECT_USE_OBJC
        #define QMapDBDispatchQueueRelease(__v)
    #else
        #define QMapDBDispatchQueueRelease(__v) (dispatch_release(__v));
    #endif
#endif

#if !__has_feature(objc_instancetype)
    #define instancetype id
#endif

typedef int(^QMapDBExecuteStatementsCallbackBlock)(NSDictionary *resultsDictionary);

// SQLite 数据库的检查点模式
typedef NS_ENUM(int, QMapDBCheckpointMode) {
    QMapDBCheckpointModePassive  = 0, // SQLITE_CHECKPOINT_PASSIVE,
    QMapDBCheckpointModeFull     = 1, // SQLITE_CHECKPOINT_FULL,
    QMapDBCheckpointModeRestart  = 2, // SQLITE_CHECKPOINT_RESTART,
    QMapDBCheckpointModeTruncate = 3  // SQLITE_CHECKPOINT_TRUNCATE
};

#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wobjc-interface-ivars"

@interface QMapDatabase : NSObject
@property (atomic, assign) BOOL traceExecution;
@property (atomic, assign) BOOL checkedOut;
@property (atomic, assign) BOOL crashOnErrors;
@property (atomic, assign) BOOL logsErrors;

@property (atomic, retain, nullable) NSMutableDictionary *cachedStatements;

+ (instancetype)databaseWithPath:(NSString * _Nullable)inPath;

+ (instancetype)databaseWithURL:(NSURL * _Nullable)url;

- (instancetype)initWithPath:(NSString * _Nullable)path;

- (instancetype)initWithURL:(NSURL * _Nullable)url;

@property (nonatomic) BOOL isOpen;
- (BOOL)open;
- (BOOL)openWithFlags:(int)flags;
- (BOOL)openWithFlags:(int)flags vfs:(NSString * _Nullable)vfsName;
- (BOOL)close;
@property (nonatomic, readonly) BOOL goodConnection;

- (BOOL)executeUpdate:(NSString*)sql withErrorAndBindings:(NSError * _Nullable __autoreleasing *)outErr, ...;
- (BOOL)update:(NSString*)sql withErrorAndBindings:(NSError * _Nullable __autoreleasing *)outErr, ...  __deprecated_msg("Use executeUpdate:withErrorAndBindings: instead");;
- (BOOL)executeUpdate:(NSString*)sql, ...;
- (BOOL)executeUpdateWithFormat:(NSString *)format, ... NS_FORMAT_FUNCTION(1,2);
- (BOOL)executeUpdate:(NSString*)sql withArgumentsInArray:(NSArray *)arguments;
- (BOOL)executeUpdate:(NSString*)sql values:(NSArray * _Nullable)values error:(NSError * _Nullable __autoreleasing *)error;
- (BOOL)executeUpdate:(NSString*)sql withParameterDictionary:(NSDictionary *)arguments;
- (BOOL)executeUpdate:(NSString*)sql withVAList: (va_list)args;
- (BOOL)executeStatements:(NSString *)sql;
- (BOOL)executeStatements:(NSString *)sql withResultBlock:(__attribute__((noescape)) QMapDBExecuteStatementsCallbackBlock _Nullable)block;

@property (nonatomic, readonly) int64_t lastInsertRowId;
@property (nonatomic, readonly) int changes;

- (QMapResultSet * _Nullable)executeQuery:(NSString*)sql, ...;
- (QMapResultSet * _Nullable)executeQueryWithFormat:(NSString*)format, ... NS_FORMAT_FUNCTION(1,2);
- (QMapResultSet * _Nullable)executeQuery:(NSString *)sql withArgumentsInArray:(NSArray *)arguments;
- (QMapResultSet * _Nullable)executeQuery:(NSString *)sql values:(NSArray * _Nullable)values error:(NSError * _Nullable __autoreleasing *)error;
- (QMapResultSet * _Nullable)executeQuery:(NSString *)sql withParameterDictionary:(NSDictionary * _Nullable)arguments;
- (QMapResultSet * _Nullable)executeQuery:(NSString *)sql withVAList:(va_list)args;
- (QMapResultSet *)prepare:(NSString *)sql;
- (BOOL)beginTransaction;
- (BOOL)beginDeferredTransaction;
- (BOOL)beginImmediateTransaction;
- (BOOL)beginExclusiveTransaction;
- (BOOL)commit;
- (BOOL)rollback;

@property (nonatomic, readonly) BOOL isInTransaction;

- (BOOL)inTransaction __deprecated_msg("Use isInTransaction property instead");
- (void)clearCachedStatements;
- (void)closeOpenResultSets;
@property (nonatomic, readonly) BOOL hasOpenResultSets;
@property (nonatomic) BOOL shouldCacheStatements;

- (BOOL)interrupt;
- (BOOL)setKey:(NSString*)key;
- (BOOL)rekey:(NSString*)key;
- (BOOL)setKeyWithData:(NSData *)keyData;
- (BOOL)rekeyWithData:(NSData *)keyData;

@property (nonatomic, readonly, nullable) NSString *databasePath;
@property (nonatomic, readonly, nullable) NSURL *databaseURL;
@property (nonatomic, readonly) void *sqliteHandle;


- (NSString*)lastErrorMessage;

- (int)lastErrorCode;

- (int)lastExtendedErrorCode;


- (BOOL)hadError;


- (NSError *)lastError;


@property (nonatomic) NSTimeInterval maxBusyRetryTimeInterval;


- (BOOL)startSavePointWithName:(NSString*)name error:(NSError * _Nullable __autoreleasing *)outErr;

- (BOOL)releaseSavePointWithName:(NSString*)name error:(NSError * _Nullable __autoreleasing *)outErr;

- (BOOL)rollbackToSavePointWithName:(NSString*)name error:(NSError * _Nullable __autoreleasing *)outErr;

- (NSError * _Nullable)inSavePoint:(__attribute__((noescape)) void (^)(BOOL *rollback))block;


- (BOOL)checkpoint:(QMapDBCheckpointMode)checkpointMode error:(NSError * _Nullable *)error;

- (BOOL)checkpoint:(QMapDBCheckpointMode)checkpointMode name:(NSString * _Nullable)name error:(NSError * _Nullable *)error;

- (BOOL)checkpoint:(QMapDBCheckpointMode)checkpointMode name:(NSString * _Nullable)name logFrameCount:(int * _Nullable)logFrameCount checkpointCount:(int * _Nullable)checkpointCount error:(NSError * _Nullable *)error;

+ (BOOL)isSQLiteThreadSafe;


- (int)limitFor:(int)type value:(int)newLimit;

+ (NSString*)sqliteLibVersion;

+ (NSString*)QMapDBUserVersion;

+ (SInt32)QMapDBVersion __deprecated_msg("Use QMapDBUserVersion instead");

- (void)makeFunctionNamed:(NSString *)name arguments:(int)arguments block:(void (^)(void *context, int argc, void * _Nonnull * _Nonnull argv))block;

- (void)makeFunctionNamed:(NSString *)name maximumArguments:(int)count withBlock:(void (^)(void *context, int argc, void * _Nonnull * _Nonnull argv))block __deprecated_msg("Use makeFunctionNamed:arguments:block:");

- (SqliteValueType)valueType:(void *)argv;

- (int)valueInt:(void *)value;

- (long long)valueLong:(void *)value;

- (double)valueDouble:(void *)value;

- (NSData * _Nullable)valueData:(void *)value;

- (NSString * _Nullable)valueString:(void *)value;

- (void)resultNullInContext:(void *)context NS_SWIFT_NAME(resultNull(context:));

- (void)resultInt:(int) value context:(void *)context;

- (void)resultLong:(long long)value context:(void *)context;

- (void)resultDouble:(double)value context:(void *)context;

- (void)resultData:(NSData *)data context:(void *)context;

- (void)resultString:(NSString *)value context:(void *)context;

- (void)resultError:(NSString *)error context:(void *)context;
- (void)resultErrorCode:(int)errorCode context:(void *)context;
- (void)resultErrorNoMemoryInContext:(void *)context NS_SWIFT_NAME(resultErrorNoMemory(context:));
- (void)resultErrorTooBigInContext:(void *)context NS_SWIFT_NAME(resultErrorTooBig(context:));

+ (NSDateFormatter *)storeableDateFormat:(NSString *)format;
- (BOOL)hasDateFormatter;
- (void)setDateFormat:(NSDateFormatter * _Nullable)format;
- (NSDate * _Nullable)dateFromString:(NSString *)s;
- (NSString * _Nullable)stringFromDate:(NSDate *)date;
@end

@interface QMapStatement : NSObject {
    void *_statement;
    NSString *_query;
    long _useCount;
    BOOL _inUse;
}

@property (atomic, assign) long useCount;
@property (atomic, retain) NSString *query;
@property (atomic, assign) void *statement;
@property (atomic, assign) BOOL inUse;

- (void)close;
- (void)reset;

@end
#pragma clang diagnostic pop
NS_ASSUME_NONNULL_END
