//
//  QTCObjcTypeAutoCoding.h
//  QMapKit
//
//  Created by tabsong on 2023/4/19.
//  Copyright © 2023 tencent. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <objc/runtime.h>

NS_ASSUME_NONNULL_BEGIN

/**
 *  @brief 不支持NSCoding的数据类型，对应的包装器协议。
 *
 *     例如：CLLocationCoordinate2D包装器 实现
 *
 *    - (nullable id)qtc_encodeWithValue:(id)value
 *    {
 *        NSValue *theValue = (NSValue *)value;
 *
 *        CLLocationCoordinate2D coordinate;
 *        [theValue getValue:(void *)&coordinate];
 *
 *        self.latitude  = coordinate.latitude;
 *        self.longitude = coordinate.longitude;
 *
 *        return self;
 *    }
 *
 *    - (nullable id)qtc_decode
 *    {
 *        CLLocationCoordinate2D coordinate = CLLocationCoordinate2DMake(self.latitude, self.longitude);
 *        return [NSValue valueWithBytes:&coordinate objCType:@encode(CLLocationCoordinate2D)];
 *    }
 *
 *    + (NSString *)qtc_supportedObjcType
 *    {
 *        return @(@encode(CLLocationCoordinate2D));
 *    }
 *
 */
@protocol QTCObjcTypeAutoCoding <NSObject>

@required

/**
 *  @brief encode 入口
 *
 *  @param value typeEncode对应的系统类，如 struct 对应 NSValue，int/float 对应NSNumber
 *  @return 根据value返回自定义的包装对象。
 */
- (nullable id)qtc_encodeWithValue:(id)value;

/**
 *  @brief decode 入口
 *
 *  @return typeEncode对应的系统类
 */
- (nullable id)qtc_decode;

/**
 *  @brief 包装器支持的 typeEncode
 *
 *   iOS TypeEncodings: https://developer.apple.com/library/archive/documentation/Cocoa/Conceptual/ObjCRuntimeGuide/Articles/ocrtTypeEncodings.html
 */
+ (NSString *)qtc_supportedObjcType;

@end

NS_ASSUME_NONNULL_END
