//
//  TNKDriveNavManager.h
//  TencentNavKit
//
//  Created by mol on 2022/7/26.
//  Copyright © 2022 Tencent. All rights reserved.
//

#import <TencentNavKit/TNKBaseNavManager.h>
#import <TencentNavKit/TNKDriveCommonObj.h>
#import <TencentNavKit/TNKDriveNavDelegate.h>
#import <TencentNavKit/TNKSearchTask.h>
#import <TencentNavKit/TNKDriveNavDataSource.h>

NS_ASSUME_NONNULL_BEGIN

/**
 * @brief 途经点移除策略
 */
typedef enum _TNKWaypointRemovedStrategy
{
    /// 默认策略。到达途经点后SDK自动判达。
    TNKWaypointRemovedStrategyDefault       = 0,
    /// 开发者移除策略。达到途经点如果想要移除该途经点，需要开发者调用changeWayPoints方法重新算路
    TNKWaypointRemovedStrategyManually      = 1,
} TNKWaypointRemovedStrategy;


/// 驾车导航管理类
@interface TNKDriveNavManager : TNKBaseNavManager

/// 单例模式，获取导航管理对象
+ (instancetype)sharedInstance;

/**
 *  @brief 是否开启导航过程中的备选路线. 默认关闭.
 */
@property (nonatomic, assign, getter=isMultiRoutesEnabled) BOOL multiRoutesEnabled;

/**
 *  @brief 开启导航时是否展示初始算路的备选路线。默认为NO
 *  @since 5.4.2.8
 */
@property (nonatomic, assign) BOOL showMultiRoutesWhenStartNav;

/**
 * @brief 导航播报模式，默认为标准模式.
 * @since 5.3.9
 */
@property (nonatomic, assign) TNKNavTTSMode ttsMode;


/**
 * @brief 设置路径规划缓存数 [1, 8]，默认 1.
 * @since 6.7.5
 */
@property (nonatomic, assign) int routePlanCacheSize;

/**
 *  @brief 当前正在导航时的主路线数据。非导航时为nil
 */
@property (nonatomic, readonly, nullable) TNKDriveNavRoute *curNavRoute;

/**
 *  @brief 当前正在导航时的伴随路线数据。非导航时为空
 */
@property (nonatomic, readonly, nullable) NSArray<TNKDriveNavRoute *> *backupNavRoutes;

/**
 * @brief 途经点移除策略 需要在开启导航前设置
 * @since 5.4.0
 */
@property (nonatomic, assign) TNKWaypointRemovedStrategy waypointRemovedStrategy;

/**
 * @brief 导航事件DataSource. since 5.2.6
 */
@property (nonatomic, weak, nullable) id<TNKDriveNavDataSource> navDataSource;

/**
 * @brief 注册对象接收导航事件变化.
 *  注册后可通过该协议获取导航过程中出现的状态或信息改变(如已偏航、经过途经点、到达终点等).
 *  导航事件变化采用广播形式,可同时注册多个对象接收事件变化.
 * @warning 内部对navDelegate弱持有
 * @param navDelegate 接收导航事件变化的对象
 */
- (void)registerNavDelegate:(id<TNKDriveNavDelegate>)navDelegate;

/**
 * @brief 注销对象接收导航事件变化.
 * @param navDelegate 接收导航事件变化的对象
 */
- (void)unregisterNavDelegate:(id<TNKDriveNavDelegate>)navDelegate;

/**
 * @brief 注销所有对象接收导航事件变化.
 */
- (void)unregisterAllNavDelegates;

/**
 * @brief 发起驾车导航路线规划. 通过路线规划请求配置起点、终点、途经点(可选)、规划参数(可选)发起路线规划.
 * @param request 路线规划请求
 * @param completion 路线规划完成回调
 *
 * @return 路线规划任务
 *
 * 错误信息参考TNKDriveRouteSearchErrorCode, TNKDriveRouteSearchErrorDomain
 */
- (TNKSearchTask * _Nullable)searchRoutesWithRequest:(TNKDriveRouteRequest *)request
                                          completion:(void (^)(TNKDriveRouteSearchResult * _Nullable searchResult,
                                                               NSError * _Nullable error))completion;

/**
 * @brief 导航过程中切换至备选路线.
 * @since 5.2.6
 * @param routeID 要切换到的路线id
 * @return 状态码。 0:成功； 1: 此路线已经是导航主路线；2:没找到对应的路线；3: 当前非导航状态，无法切换
 */
- (int)changeToBackupRoute:(NSString *)routeID;

/**
 * @brief 导航过程中删除某备选路线.
 * @since 5.2.6
 * @param routeID 路线id
 * @return 状态码。 0:成功； 1: 此路线已经是导航主路线；2:没找到对应的路线；3: 当前非导航状态，无法删除
 */
- (int)deleteBackupRoute:(NSString *)routeID;

/**
 * @brief 导航过程中删除所有备选路线
 * @since 6.0.0
 */
- (void)deleteAllBackupRoutes;

/**
 * @brief 导航过程中增加轮次播报回放。导航过程中有效，当前正在播报时，调用此方法可能无效。
 * @since 5.3.9
 */
- (void)requestTts;

/**
 * @brief 设置终点偏航参数。注意:需要导航前设置才能生效。参数有效条件:(destArriveRadius >= 30 && destArriveRadius <= destOffRouteRadius)
 * @since 6.7.0
 * @param destArriveRadius 进入半径，单位,米
 * @param destOffRouteRadius 驶离半径，单位,米
 */
- (void)setDestArriveOffRouteRadius:(int)destArriveRadius destOffRouteRadius:(int)destOffRouteRadius;

@end

NS_ASSUME_NONNULL_END
