//
//  TencentLBSGeoFenceManager.h
//  TencentLBS
//
//  Created by Ranyruan on 2023/5/19.
//  Copyright © 2023 Tencent. All rights reserved.
//

#import <Foundation/Foundation.h>
#import "TencentLBSGeoFenceRegion.h"

NS_ASSUME_NONNULL_BEGIN

@protocol TencentLBSGeoFenceManagerDelegate;


@interface TencentLBSGeoFenceManager : NSObject

///实现了 TencentLBSGeoFenceManagerDelegate 协议的类指针。
@property (nonatomic, weak) id<TencentLBSGeoFenceManagerDelegate> delegate;

/**
 * @brief 添加一个圆形围栏
 * @param center 围栏的中心点经纬度坐标
 * @param radius 围栏的半径，单位：米，要求大于0
 * @param customID 用户自定义ID，可选，SDK原值返回
 */
- (void)addCircleRegionForMonitoringWithCenter:(CLLocationCoordinate2D)center radius:(CLLocationDistance)radius customID:(NSString *)customID;


/**
 * @brief 根据经纬度坐标数据添加一个闭合的多边形围栏，点与点之间按顺序尾部相连, 第一个点与最后一个点相连
 * @param coordinates 经纬度坐标点数据,coordinates对应的内存会拷贝,调用者负责该内存的释放
 * @param count 经纬度坐标点的个数，不可小于3个
 * @param customID 用户自定义ID，可选，SDK原值返回
 */
- (void)addPolygonRegionForMonitoringWithCoordinates:(CLLocationCoordinate2D *)coordinates count:(NSInteger)count customID:(NSString *)customID;


/**
 * @brief 获取指定围栏的运行状态
 * @param region 要获取运行状态的围栏
 * @return 返回指定围栏的运行状态
 */
- (TencentLBSGeoFenceRegionActiveStatus)statusWithGeoFenceRegion:(TencentLBSGeoFenceRegion *)region;

/**
 * @brief 根据customID获得所有已经注册的围栏，如果customID传nil，则返回全部已注册围栏
 * @param customID 用户执行添加围栏函数时传入的customID
 * @return 获得的围栏构成的数组，如果没有结果，返回nil
 */
- (NSArray *)geoFenceRegionsWithCustomID:(NSString *)customID;

/**
 * @brief 根据customID获得所有正在监控的围栏，如果customID传nil，则返回全部正在监控的围栏
 * @param customID 用户执行添加围栏函数时传入的customID
 * @return 获得的围栏构成的数组，如果没有结果，返回nil
 */
- (NSArray *)monitoringGeoFenceRegionsWithCustomID:(NSString *)customID;

/**
 * @brief 根据customID获得所有已经暂停的围栏，如果customID传nil，则返回全部已经暂停的围栏
 * @param customID 用户执行添加围栏函数时传入的customID
 * @return 获得的围栏构成的数组，如果没有结果，返回nil
 */
- (NSArray *)pausedGeoFenceRegionsWithCustomID:(NSString *)customID;


/**
 * @brief 暂停指定customID的围栏
 * @param customID 用户执行添加围栏函数时传入的customID
 * @return 返回被暂停围栏的数组，如果没有围栏被暂停，返回nil
 */
- (NSArray *)pauseGeoFenceRegionsWithCustomID:(NSString *)customID;


/**
 * @brief 暂停指定围栏
 * @param region 要暂停监控的围栏
 * @return 返回指定围栏是否被暂停，如果指定围栏没有注册，则返回NO
 */
- (BOOL)pauseTheGeoFenceRegion:(TencentLBSGeoFenceRegion *)region;

/**
 * @brief 根据customID开始监控已经暂停的围栏
 * @param customID 用户执行添加围栏函数时传入的customID
 * @return 返回开始监控的围栏构成的数组
 */
- (NSArray *)startGeoFenceRegionsWithCustomID:(NSString *)customID;

/**
 * @brief 开始监控指定围栏
 * @param region 要开始监控的围栏
 * @return 返回指定围栏是否开始监控，如果指定围栏没有注册，则返回NO
 */
- (BOOL)startTheGeoFenceRegion:(TencentLBSGeoFenceRegion *)region;

/**
 * @brief 移除指定围栏
 * @param region 要停止监控的围栏
 */
- (void)removeTheGeoFenceRegion:(TencentLBSGeoFenceRegion *)region;

/**
 * @brief 移除指定customID的围栏
 * @param customID 用户执行添加围栏函数时传入的customID
 */
- (void)removeGeoFenceRegionsWithCustomID:(NSString *)customID;

/**
 * @brief 移除所有围栏
 */
- (void)removeAllGeoFenceRegions;

@end

@protocol TencentLBSGeoFenceManagerDelegate <NSObject>

@optional

/**
 * @brief 添加地理围栏完成后的回调，成功与失败都会调用
 * @param manager 地理围栏管理类
 * @param action 定位SDK相关的状态
 */
- (void)TencentLBSGeoFenceManager:(TencentLBSGeoFenceManager *)manager
                locationSDKAction:(TencentLBSLocationSDKAction)action;


/**
 * @brief 地理围栏状态改变时回调，当围栏状态的值发生改变，定位失败都会调用
 * @param manager 地理围栏管理类
 * @param region 状态改变的地理围栏
 * @param customID 用户执行添加围栏函数时传入的customID
 */
- (void)TencentLBSGeoFenceManager:(TencentLBSGeoFenceManager *)manager didGeoFencesStatusChangedForRegion:(TencentLBSGeoFenceRegion *)region customID:(NSString *)customID;

@end
NS_ASSUME_NONNULL_END
